// Title      : Sample.java
// Author     : James Baird
// Created    : Saturday, 28th April 2001
// Description: Sample Class

import com.oroinc.net.nntp.NewsgroupInfo;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.HashMap;
import org.apache.xerces.dom.DocumentImpl;
import org.apache.xml.serialize.XMLSerializer;
import org.apache.xml.serialize.OutputFormat;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;
import java.util.StringTokenizer;
import java.util.Vector;

public class Sample
{
  public static void main(String[] args)
  {
    if (args.length < 1)
    {
      System.err.println("Usage:");
      System.err.println("  Sample snapshot <outfile>");
      System.err.println("  Sample groups <infile> <outfile> <percent size>");
      System.err.println("  Sample content <infile> <outfile> <size>");
      System.err.println("  Sample catalog <infile> <outfile>");
      System.err.println("  Sample days <infile> <start day> <end day> <start group>");
    }
    else
    {
      if (args[0].equals("snapshot"))
        sampleSnapshot("news.btinternet.com", args[1]);
      else if (args[0].equals("groups"))
        sampleGroups("news.btinternet.com", args[1], args[2], Integer.parseInt(args[3]) / 100.0);
      else if (args[0].equals("content"))
        sampleContent("news.btinternet.com", args[1], args[2], Integer.parseInt(args[3]));
      else if (args[0].equals("catalog"))
        sampleCatalog(args[1], args[2]);
      else if (args[0].equals("days"))
        sampleDays("news.btinternet.com", args[1], Integer.parseInt(args[2]), Integer.parseInt(args[3]), Integer.parseInt(args[4]));
    }
  }
  
  protected static void sampleSnapshot(String server, String outfilename)
  {
    System.out.println("Snapshot from " + server + " into file " + outfilename);
    
    NewsClient client = new NewsClient();
    
    try
    {
      NewsgroupInfo[] newsgroups;
      
      client.connect(server);
      
      newsgroups = client.listNewsgroups();
      
      if (newsgroups != null)
      {
        FileWriter outfile = new FileWriter(outfilename);

        for (int i = 0; i < newsgroups.length; i++)
          outfile.write(newsgroups[i].getNewsgroup() + " " + newsgroups[i].getArticleCount() + System.getProperty("line.separator"));
        
        outfile.flush();
        outfile.close();
      }
      
      client.logout();
    }
    catch (IOException e)
    {
      e.printStackTrace();
    }
    finally
    {
      try
      {
        client.disconnect();
      }
      catch (IOException e)
      {
        e.printStackTrace();
      }
    }
  }
  
  protected static void sampleGroups(String server, String infilename, String outfilename, double size)
  {
    System.out.println(size * 100 + "% group sample from file " + infilename + " to file " + outfilename);
    
    NewsClient client = new NewsClient();

    try
    {
      System.out.println("Reading index...");

      Vector index = importIndex(infilename);
      
      System.out.println(index.size() + " group(s) read");

      client.connect(server);

      Vector selection = new Vector();
      
      System.out.println("Creating sample...");
      
      for (int i = 0; i < index.size() * size; i++)
      {
        Group group;
        
        do
        {
          do
          {
            int g;
            
            do
            {
              g = (int)(Math.random() * index.size());
          
              group = (Group)index.elementAt(g);
            }
            while (group == null);
            
            index.setElementAt(null, g);
          }
          while (!group.acceptable() || selection.indexOf(group) != -1);
        }
        while (!group.testSize(client, 64));
        
        selection.addElement(group);
      }

      System.out.println("Writing results...");
      
      FileWriter outfile = new FileWriter(outfilename);
      
      for (int i = 0; i < selection.size(); i++)
      {
        Group group = (Group)selection.elementAt(i);
        
        outfile.write(group.getName() + " " + group.getCount() + System.getProperty("line.separator"));
      }
      
      outfile.flush();
      outfile.close();
      
      client.logout();
    }
    catch (IOException e)
    {
      e.printStackTrace();
    }
    finally
    {
      try
      {
        client.disconnect();
      }
      catch (IOException e)
      {
        e.printStackTrace();
      }
    }
  }
  
  protected static void sampleContent(String server, String infilename, String outfilename, int size)
  {
    System.out.println(size + " post sample from server " + server + " and file " + infilename + " to file " + outfilename);

    NewsClient client = new NewsClient();
    
    try
    {
      client.connect(server);
      
      System.out.println("Reading groups...");

      Vector index = importIndex(infilename);

      System.out.println(index.size() + " group(s) read");
      
      FileWriter outfile = new FileWriter(outfilename);
      
      System.out.println("Accumulating posts...");

      for (int i = 0; i < size; i++)
      {
        Vector header;
        Vector body;
        int g;
        Group group;
        int post;
      
        do
        {
          do
          {
            g = (int)(Math.random() * index.size());
          
            group = (Group)index.elementAt(g);
          }
          while (group == null);
        
          NewsgroupInfo info = new NewsgroupInfo();
        
          client.selectNewsgroup(group.getName(), info);
        
          post = (int)(Math.random() * group.getCount()) + info.getFirstArticle();
            
          header = client.getPostHeader(post);
          body = client.getPostBody(post);
        }
        while (header == null || body == null);
        
        index.setElementAt(null, g);
        
        appendPost(outfile, group.getName(), post, header, body);
      }
      
      outfile.flush();
      outfile.close();
      
      client.logout();
    }
    catch (IOException e)
    {
      e.printStackTrace();
    }
    finally
    {
      try
      {
        client.disconnect();
      }
      catch (IOException e)
      {
        e.printStackTrace();
      }
    }
  }
  
  protected static void sampleCatalog(String infilename, String outfilename)
  {
    System.out.println("Catalogue from file " + infilename + " to file " + outfilename);

    try
    {
      System.out.println("Reading index...");
    
      Vector index = importIndex(infilename);
    
      System.out.println(index.size() + " group(s) read");
    
      FileWriter outfile = new FileWriter(outfilename);
    
      for (int i = 0; i < index.size(); i++)
      {
        Group group = (Group)index.elementAt(i);
      
        outfile.write(group.getName() + " " + group.getCount() + System.getProperty("line.separator"));
      }
    
      outfile.flush();
      outfile.close();
    }
    catch (IOException e)
    {
      e.printStackTrace();
    }
  }
  
  protected static void sampleDays(String server, String infilename, int firstDay, int lastDay, int startGroup)
  {
    System.out.println("days sample (start " + firstDay + ", end " + lastDay + ") from server " + server + " and file " + infilename);

    NewsClient client = new NewsClient();
    
    try
    {
      client.connect(server);
      
      System.out.println("Reading groups...");

      Vector index = importIndex(infilename);

      System.out.println(index.size() + " group(s) read");

      Calendar today = new GregorianCalendar();

      Calendar firstDate = new GregorianCalendar();

      firstDate.clear();

      firstDate.set(today.get(Calendar.YEAR), today.get(Calendar.MONTH), firstDay, 0, 0, 0);

      Calendar lastDate = new GregorianCalendar();

      lastDate.clear();

      lastDate.set(today.get(Calendar.YEAR), today.get(Calendar.MONTH), lastDay, 0, 0, 0);
      
      System.out.println("Accumulating posts...");

      HashMap used = new HashMap();

      for (int i = startGroup; i < index.size(); i++)
      {
        Group group;

        group = (Group)index.elementAt(i);

        System.out.println(i + " : " + group.getName());

        Vector posts = group.getPosts(client, firstDate, lastDate);

        if (posts != null)
        {
          System.out.println(posts.size() + " post(s) downloaded");

          Document document = new DocumentImpl();

          Element groupElement = document.createElement("group");
          groupElement.setAttribute("name", group.getName());
          groupElement.setAttribute("volume", Integer.toString(posts.size()));

          document.appendChild(groupElement);

          int id = 0;

          for (int j = posts.size() - 1; j >= 0; j--)
          {
            Post post = (Post)posts.elementAt(j);

            Element postElement = post.toElement(document);
            postElement.setAttribute("id", Integer.toString(id++));

            groupElement.appendChild(postElement);
          }
          
          String filename = "logs/" + group.getName() + ".xml";

          System.out.println("Exporting group");

          export(filename, document);
        }
      }

      client.logout();
    }
    catch (IOException e)
    {
      e.printStackTrace();
    }
    finally
    {
      try
      {
        client.disconnect();
      }
      catch (IOException e)
      {
        e.printStackTrace();
      }
    }
  }

  protected static Vector importIndex(String infilename) throws IOException
  {
    Vector index = new Vector();
    
    BufferedReader infile = new BufferedReader(new FileReader(infilename));
    
    String indexLine = infile.readLine();
    while (indexLine != null)
    {
      if (indexLine.equals("eof"))
        break;
    
      if (!indexLine.equals("") && indexLine.indexOf("*") == -1)
      {
        StringTokenizer tokeniser = new StringTokenizer(indexLine);

        index.addElement(new Group(tokeniser.nextToken(), Integer.parseInt(tokeniser.nextToken())));
      }
    
      indexLine = infile.readLine();
    }
    
    infile.close();
    
    return index;
  }

  protected static void exportVector(FileWriter outfile, Vector vector) throws IOException
  {
    if (vector != null)
      for (int i = 0; i < vector.size(); i++)
        outfile.write((String)vector.elementAt(i) + System.getProperty("line.separator"));
  }
  
  protected static void appendPost(FileWriter outfile, String groupName, int articleNumber, Vector header, Vector body) throws IOException
  {
    outfile.write("*** " + groupName + " : " + articleNumber + System.getProperty("line.separator"));
    
    if (header != null && body != null)
    {
      exportVector(outfile, header);
      
      outfile.write("---" + System.getProperty("line.separator"));
      
      exportVector(outfile, body);
      
      outfile.write(System.getProperty("line.separator"));
    }
  }

  protected static void export(String filename, Document document) throws IOException
  {
    BufferedWriter bw = new BufferedWriter(new FileWriter(filename));

    XMLSerializer serialiser = new XMLSerializer(bw, new OutputFormat(document));

    serialiser.serialize(document);

    bw.flush();
    bw.close();
  }
}
